<?php

namespace IPS\heleket;

use DomainException;
use Exception;
use InvalidArgumentException;
use IPS\DateTime;
use IPS\Helpers\Form;
use IPS\Helpers\Form\Text;
use IPS\Http\Url;
use IPS\nexus\Customer\CreditCard;
use IPS\nexus\Fraud\MaxMind\Request;
use IPS\nexus\Gateway;
use IPS\nexus\Transaction;
use IPS\Settings;
use LogicException;
use function defined;

/* To prevent PHP errors (extending class does not exist) revealing path */
if (!defined('\IPS\SUITE_UNIQUE_KEY'))
{
	header(($_SERVER['SERVER_PROTOCOL'] ?? 'HTTP/1.0') . ' 403 Forbidden');
	exit;
}

/**
 * Heleket Payment Gateway
 */
class _Heleket extends Gateway
{

	/**
	 * @brief    Database Table
	 */
	public static $scopes = 'account-info operation-history payment-p2p';

	/* !Features (Each gateway will override) */

	const SUPPORTS_REFUNDS = false;
	const SUPPORTS_PARTIAL_REFUNDS = false;

	/* !Payment Gateway */

	/**
	 * Authorize
	 *
	 * @param Transaction      $transaction                      Transaction
	 * @param array|CreditCard $values                           Values from form OR a stored card object if this
	 *                                                           gateway supports them
	 * @param Request|NULL     $maxMind                          *If* MaxMind is enabled, the request object will be
	 *                                                           passed here so gateway can additional data before
	 *                                                           request is made
	 *
	 * @return    DateTime|NULL        Auth is valid until or NULL to indicate auth is good forever
	 * @throws    LogicException            Message will be displayed to user
	 */
	public function auth(
		Transaction $transaction,
					$values,
		Request     $maxMind = null,
					$recurrings = [],
					$source = null
	)
	{
		$transaction->save();

		$params = [
			'amount'       => $transaction->amount->amountAsString(),
			'currency'     => $transaction->amount->currency,
			'order_id'     => (string) $transaction->id,
			'url_return'   => (string) $transaction->invoice->url(),
			'url_callback' => Settings::i()->base_url . '?app=heleket&module=system&controller=callback&notify=1'
		];

		$settings = json_decode($this->settings, true);

		$invoice = $this->apiCall(
			$settings['merchant_id'],
			$settings['api_key'],
			'v1/payment',
			$params,
			$error
		);
		if ($error)
		{
			throw new DomainException($error);
		}

		return \IPS\Output::i()->redirect(Url::external($invoice['url']));
	}

	public function getApiEndpoint(): string
	{
		return 'https://api.heleket.com/';
	}

	public function getSignature(string $data, string $apiKey): string
	{
		return hash('md5', base64_encode($data) . $apiKey);
	}

	private function apiCall(
		string $merchantId,
		string $apiKey,
		string $path,
		array  $data,
			   &$error = null
	): array
	{
		$dataString = json_encode($data, JSON_UNESCAPED_UNICODE);
		$headers = [
			'Content-type' => 'application/json',
			'merchant'     => $merchantId,
			'sign'         => $this->getSignature($dataString, $apiKey),
			'Content-Type' => 'application/json'
		];

		try
		{
			$response = Url::external($this->getApiEndpoint() . $path)
						   ->request()
						   ->setHeaders($headers)
						   ->post($dataString);
		}
		catch (Exception $e)
		{
			$error = $e->getMessage();

			return [];
		}

		$responseData = json_decode($response->content, true);

		$state = $responseData['state'] ?? 0;
		if (is_array($responseData) && $state == 0)
		{
			return $responseData['result'] ?? [];
		}
		else
		{
			if (!empty($responseData['message']))
			{
				$error = $responseData['message'];

				return [];
			}

			if (!empty($responseData['errors']))
			{
				$error = is_array($responseData['errors'])
					? json_encode($responseData['errors'])
					: $responseData['errors'];

				return [];
			}
		}

		$error = 'something_went_wrong_please_try_again';

		return [];
	}

	/* !ACP Configuration */

	/**
	 * Settings
	 *
	 * @param Form $form The form
	 *
	 * @return    void
	 */
	public function settings(&$form)
	{
		$settings = json_decode($this->settings, true);

		$form->add(new Text('heleket_merchant_id', $settings['merchant_id'] ?? null, true));
		$form->add(new Text('heleket_api_key', $settings['api_key'] ?? null, true));
	}

	/**
	 * Test Settings
	 *
	 * @return    void
	 * @throws    InvalidArgumentException
	 */
	public function testSettings($settings)
	{
		return $settings;
	}

}
