<?php

use GuzzleHttp\Exception\GuzzleException;
use GuzzleHttp\Exception\RequestException;

if (!defined('_PS_VERSION_'))
{
	exit;
}

class Heleket extends PaymentModule
{
	private $_html = '';
	private $_postErrors = [];

	public $address;

	public function __construct()
	{
		$this->name = 'heleket';
		$this->tab = 'payments_gateways';
		$this->version = '1.0.0';
		$this->author = 'TeslaCloud Studios';
		$this->need_instance = 0;
		$this->ps_versions_compliancy = [
			'min' => '8.0',
			'max' => '8.99.99',
		];
		$this->controllers = ['payment', 'validation'];

		$this->bootstrap = true;

		parent::__construct();

		$this->displayName = $this->trans('Heleket', [], 'Modules.Heleket.Admin');
		$this->description = $this->trans('Heleket payment profile for PrestaShop.', [], 'Modules.Heleket.Admin');

		$this->confirmUninstall = $this->trans('Are you sure you want to uninstall?', [], 'Modules.Heleket.Admin');

		if (!Configuration::get('heleket'))
		{
			$this->warning = $this->l('No name provided');
		}

		return $this->warning;
	}

	public function install()
	{
		return parent::install()
			&& $this->registerHook('paymentOptions')
			&& $this->registerHook('paymentReturn')
			&& $this->registerHook('orderConfirmation')
			&& $this->installOrderState();
	}

	public function uninstall()
	{
		return parent::uninstall() && Configuration::deleteByName('heleket');
	}

	private function installOrderState()
	{
		$order_state = new OrderState();
		$order_state->name = [];

		foreach (Language::getLanguages() as $language)
		{
			$order_state->name[$language['id_lang']] = 'Awaiting for Heleket payment';
		}
		$order_state->send_email = false;
		$order_state->color = '#4169E1';
		$order_state->hidden = false;
		$order_state->delivery = false;
		$order_state->logable = false;
		$order_state->invoice = false;
		$order_state->module_name = $this->name;
		$order_state->template = 'payment_expected_heleket';

		try
		{
			$order_state->add();
			Configuration::updateValue('PAYMENT_EXPECTED_HELEKET', $order_state->id);

			return true;
		}
		catch (PrestaShopDatabaseException|PrestaShopException $e)
		{
			return false;
		}
	}

	public function getApiEndpoint(): string
	{
		return 'https://api.heleket.com/';
	}

	public function hookOrderConfirmation($params)
	{
	}

	protected function getPaymentParams($order)
	{
		$params = [
			'amount'     => number_format(
				$order->total_paid, 2, '.', ''
			),
			'currency'   => (new Currency($order->id_currency))->iso_code,
			'order_id'   => 'prestashop_id_' . $order->id_cart . '_code_' . $order->reference,
			'url_return' => $this->getReturnUrl($order)
		];

		return $params;
	}

	private function getReturnUrl($order)
	{
		$cart = new Cart($order->id_cart);
		$customer = new Customer($cart->id_customer);

		return Tools::getShopDomainSsl(true) . __PS_BASE_URI__ . 'index.php?controller=order-confirmation&id_cart=' . $cart->id . '&id_order=' . $order->id . '&key='
			. $customer->secure_key . '&id_module=' . $this->id;
	}

	public function hookPaymentReturn($params)
	{
		/*
		 * Verify if this module is enabled
		 */
		if (!$this->active)
		{
			$this->_errors[] = $this->trans('This payment method is not active.', [], 'Modules.Heleket.Payment');
		}

		if ($params['order']->current_state == Configuration::get('PS_OS_PAYMENT'))
		{
			return $this->trans('Order is paid', [], 'Modules.Heleket.Payment');
		}

		if ($this->checkPaymentStatus($params))
		{
			return $this->trans('Order is paid', [], 'Modules.Heleket.Payment');
		}

		$order = $params['order'];
		$paymentParams = $this->getPaymentParams($order);

		$payment = $this->apiCall(
			Configuration::getGlobalValue('HELEKET_MERCHANT_ID'),
			Configuration::getGlobalValue('HELEKET_API_KEY'),
			'v1/payment',
			$paymentParams,
			$error
		);
		if ($error)
		{
			$this->_errors[] = $error;
		}

		$textButton = Configuration::getGlobalValue('HELEKET_TEXT_BUTTON_PAY') ?? 'Pay with Heleket';

		$this->smarty->assign(['textButton' => $textButton]);
		$this->smarty->assign(['url' => $payment['url']]);

		return $this->fetch('module:heleket/views/templates/hook/payment_return.tpl');
	}

	private function checkPaymentStatus($params): bool
	{
		try
		{
			$payment = $this->apiCall(
				Configuration::getGlobalValue('HELEKET_MERCHANT_ID'),
				Configuration::getGlobalValue('HELEKET_API_KEY'),
				'v1/payment/info',
				[
					'order_id' => 'prestashop_id_' . $params['order']->id_cart . '_code_' . $params['order']->reference
				],
				$error
			);
			if ($error)
			{
				return false;
			}

			if (in_array($payment['payment_status'], ['paid', 'paid_over']))
			{
				if ($params['order']->current_state == Configuration::get('PAYMENT_EXPECTED_HELEKET'))
				{
					$objOrder = new Order($params['order']->id);
					$history = new OrderHistory();
					$history->id_order = (int) $objOrder->id;
					$history->changeIdOrderState(Configuration::get('PS_OS_PAYMENT'), (int) $objOrder->id);
				}

				return true;
			}
		}
		catch (Exception $e)
		{
		}

		return false;
	}

	public function hookPaymentOptions($params)
	{
		/*
		 * Verify if this module is active
		 */
		if (!$this->active)
		{
			return [];
		}

		/**
		 * Form action URL. The form data will be sent to the
		 * validation controller when the user finishes
		 * the order process.
		 */
		$formAction = $this->context->link->getModuleLink($this->name, 'validation', [], true);

		/*
		 * Assign the url form action to the template var $action
		 */
		$this->smarty->assign(['action' => $formAction]);

		/**
		 *  Load form template to be displayed in the checkout step
		 */
		$paymentForm = $this->fetch('module:heleket/views/templates/hook/payment_options.tpl');

		/**
		 * Create a PaymentOption object containing the necessary data
		 * to display this module in the checkout
		 */
		$paymentOption = new PrestaShop\PrestaShop\Core\Payment\PaymentOption();
		$paymentOption->setModuleName($this->displayName)
					  ->setCallToActionText($this->displayName)
					  ->setAction($formAction)
					  ->setForm($paymentForm);

		return [$paymentOption];
	}

	public function renderForm()
	{
		$formFields = [
			'form' => [
				'legend' => [
					'title' => $this->trans('Edit payment method', [], 'Modules.Heleket.Admin'),
					'icon'  => 'icon-envelope',
				],
				'input'  => [
					[
						'type'     => 'text',
						'label'    => 'Merchant ID',
						'name'     => 'HELEKET_MERCHANT_ID',
						'required' => true,
					],
					[
						'type'     => 'text',
						'label'    => 'Payment API key',
						'name'     => 'HELEKET_API_KEY',
						'required' => true,
					],
					[
						'type'     => 'text',
						'label'    => 'Pay Now Button Text: ',
						'name'     => 'HELEKET_TEXT_BUTTON_PAY',
						'required' => false,
					],
				],
				'submit' => [
					'title' => $this->trans('Save', [], 'Admin.Actions'),
				],
			],
		];

		$helper = new HelperForm();
		$helper->show_toolbar = false;
		$helper->id = (int) Tools::getValue('id_carrier');
		$helper->identifier = $this->identifier;
		$helper->submit_action = 'btnSubmit';
		$helper->currentIndex = $this->context->link->getAdminLink('AdminModules', false, [], [
				'configure'   => $this->name,
				'tab_module'  => $this->tab,
				'module_name' => $this->name
			]
		);
		$helper->token = Tools::getAdminTokenLite('AdminModules');
		$helper->tpl_vars = [
			'fields_value' => $this->getConfigFieldsValues(),
		];

		return $helper->generateForm([$formFields]);
	}

	public function getConfigFieldsValues()
	{
		return [
			'HELEKET_API_KEY'         => Tools::getValue('HELEKET_API_KEY', Configuration::get('HELEKET_API_KEY')),
			'HELEKET_MERCHANT_ID'     => Tools::getValue('HELEKET_MERCHANT_ID', Configuration::get('HELEKET_MERCHANT_ID')),
			'HELEKET_TEXT_BUTTON_PAY' => Tools::getValue('HELEKET_TEXT_BUTTON_PAY', Configuration::get('HELEKET_TEXT_BUTTON_PAY')),
		];
	}

	/**
	 * Returns a string containing the HTML necessary to
	 * generate a configuration screen on the admin
	 *
	 * @return string
	 */
	public function getContent()
	{
		$html = '';

		if (Tools::isSubmit('btnSubmit'))
		{
			$this->_postValidation();
			if (!count($this->_postErrors))
			{
				$this->_postProcess();
			}
			else
			{
				foreach ($this->_postErrors as $error)
				{
					$html .= $this->displayError($error);
				}
			}
		}

		$html .= $this->renderForm();

		return $html;
	}

	private function _postValidation()
	{
		if (Tools::isSubmit('btnSubmit'))
		{
			if (!Tools::getValue('HELEKET_MERCHANT_ID'))
			{
				$this->_postErrors[] = $this->trans('Merchant ID is required', [], 'Modules.Heleket.Admin');
			}
			if (!Tools::getValue('HELEKET_API_KEY'))
			{
				$this->_postErrors[] = $this->trans('Api key is required.', [], 'Modules.Heleket.Admin');
			}
		}
	}

	private function _postProcess()
	{
		if (Tools::isSubmit('btnSubmit'))
		{
			Configuration::updateValue('HELEKET_API_KEY', Tools::getValue('HELEKET_API_KEY'));
			Configuration::updateValue('HELEKET_MERCHANT_ID', Tools::getValue('HELEKET_MERCHANT_ID'));
			Configuration::updateValue('HELEKET_TEXT_BUTTON_PAY', Tools::getValue('HELEKET_TEXT_BUTTON_PAY'));
		}

		$this->_html .= $this->displayConfirmation($this->trans('Settings updated', [], 'Admin.Notifications.Success'));
	}

	/**
	 * @param string $merchantId
	 * @param string $apiKey
	 * @param string $path
	 * @param array  $data
	 * @param        $error
	 *
	 * @return array
	 * @throws GuzzleException
	 */
	public function apiCall(string $merchantId, string $apiKey, string $path, array $data, &$error = null): array
	{
		$dataString = json_encode($data, JSON_UNESCAPED_UNICODE);

		try
		{
			$httpClient = new GuzzleHttp\Client([
				'verify'  => true,
				'headers' => [
					'User-Agent' => 'PrestaShop/' . _PS_VERSION_,
				],
			]);

			$response = $httpClient->post($this->getApiEndpoint() . $path, [
				'headers' => [
					'merchant'     => $merchantId,
					'sign'         => $this->getSignature($dataString, $apiKey),
					'Content-Type' => 'application/json'
				],
				'body'    => $dataString
			]);
		}
		catch (RequestException $e)
		{
			if ($e->hasResponse() && $e->getResponse()->getStatusCode() != 401)
			{
				$responseData = json_decode($e->getResponse()->getBody(), true);

				$error = $responseData['message'] ?? $this->trans('Something went wrong. Please try again or contact the administrator.', [], 'Modules.Heleket.Api');

				return [];
			}

			$error = $this->trans('Something went wrong. Please try again or contact the administrator.', [], 'Modules.Heleket.Api');

			return [];
		}

		$responseData = json_decode($response->getBody(), true);

		$state = $responseData['state'] ?? 0;
		if (is_array($responseData) && $state == 0)
		{
			return $responseData['result'] ?? [];
		}
		else
		{
			if (!empty($responseData['message']))
			{
				$error = $responseData['message'];

				return [];
			}

			if (!empty($responseData['errors']))
			{
				$error = is_array($responseData['errors'])
					? json_encode($responseData['errors'])
					: $responseData['errors'];

				return [];
			}
		}

		$error = $this->trans('Something went wrong. Please try again or contact the administrator.', [], 'Modules.Heleket.Api');

		return [];
	}

	/**
	 * @param string $data
	 * @param string $apiKey
	 *
	 * @return string
	 */
	public function getSignature(string $data, string $apiKey): string
	{
		return hash('md5', base64_encode($data) . $apiKey);
	}

	/**
	 * @param $network
	 * @param $address
	 * @param $amount
	 *
	 * @return string
	 */
	public function generatePaymentLink($network, $address, $amount): string
	{
		if ($network == 'ton')
		{
			$amount = number_format($amount, 9, '', '');

			return 'https://app.tonkeeper.com/transfer/' . $address . '?amount=' . $amount;
		}

		return $network . ':' . $address . '?amount=' . $amount;
	}

	/**
	 * Список хуков, используемых в модуле.
	 *
	 * @return string[]
	 */
	private function getHooksList(): array
	{
		return [
			'paymentOptions',
			'paymentReturn',
			'orderConfirmation'
		];
	}
}