<?php

namespace Heleket\Woocommerce;

use WP_Error;

final class RequestBuilder
{
	const API_URL = "https://api.heleket.com/";

	/**
	 * @var string
	 */
	private string $secretKey;
	/**
	 * @var string
	 */
	private string $merchantUuid;

	/**
	 * @param string $secretKey
	 * @param string $merchantUuid
	 */
	public function __construct(string $secretKey, string $merchantUuid)
	{
		$this->secretKey = $secretKey;
		$this->merchantUuid = $merchantUuid;
	}

	/**
	 * @param string $uri
	 * @param array  $data
	 *
	 * @return bool|mixed
	 * @throws RequestBuilderException
	 */
	public function sendRequest(string $uri, array $data = [])
	{
		$url = self::API_URL . $uri;
		$body = json_encode($data, JSON_UNESCAPED_UNICODE);

		$headers = [
			'Accept'       => 'application/json',
			'Content-Type' => 'application/json;charset=UTF-8',
			'merchant'     => $this->merchantUuid,
			'sign'         => hash('md5', base64_encode($body) . $this->secretKey)
		];

		$response = wp_remote_post($url, [
			'headers' => $headers,
			'body'    => $body
		]);

		if ($response instanceof WP_Error)
		{
			throw new RequestBuilderException(
				esc_html($response->get_error_message()),
				esc_html($response->get_error_code()),
				esc_url($uri)
			);
		}

		$responseCode = wp_remote_retrieve_response_code($response);
		$responseJson = wp_remote_retrieve_body($response);
		if (!empty($responseJson))
		{
			$json = json_decode($responseJson, true);
			if (is_null($json))
			{
				throw new RequestBuilderException(
					esc_html(json_last_error_msg()),
					esc_html($responseCode),
					esc_url($uri)
				);
			}

			if ($responseCode !== 200 || (!is_null($json['state']) && $json['state'] != 0))
			{
				if (!empty($json['message']))
				{
					throw new RequestBuilderException(
						esc_html($json['message']),
						esc_html($responseCode),
						esc_html($uri));
				}

				if (!empty($json['errors']))
				{
					throw new RequestBuilderException(
						'Validation error',
						esc_html($responseCode),
						esc_html($uri),
						esc_html($json['errors'])
					);
				}
			}

			if (!empty($json['result']) && !is_null($json['state']) && $json['state'] == 0)
			{
				return $json['result'];
			}
		}

		return true;
	}
}
