<?php

namespace Heleket\Woocommerce;

use Exception;
use WC_Payment_Gateway;

final class Gateway extends WC_Payment_Gateway
{
	/**
	 * @var string
	 */
	public $id = 'heleket-crypto-gateway-for-woocommerce';
	/**
	 * @var bool
	 */
	public $has_fields = true;
	/**
	 * @var string
	 */
	public $title = 'Pay with Heleket (USDT, USDC, BTC, ETH, LTC, and more)';
	/**
	 * @var string
	 */
	public $method_title = 'Heleket';
	/**
	 * @var string
	 */
	public $method_description = 'Heleket payment profile for WooCommerce.';
	/**
	 * @var Payment
	 */
	public $payment;
	/**
	 * @var string
	 */
	public $merchant_uuid;
	/**
	 * @var int|string
	 */
	public $subtract;
	/**
	 * @var string
	 */
	private $payment_key;
	/**
	 * @var string
	 */
	public $lifetime;
	public $accepted_networks;
	public $accepted_currencies;
	public $h2h;
	public $theme;

	public function __construct()
	{
		$this->title = $this->get_option('method_title') ?: $this->title;
		$this->method_description = '';
		$this->description = $this->get_option('description');
		$this->form_fields = $this->adminFields();
		$this->init_settings();

		$this->payment_key = $this->get_option('payment_key');
		$this->merchant_uuid = $this->get_option('merchant_uuid');
		$this->subtract = $this->get_option('subtract') ?? 0;
		$this->payment = Client::payment($this->payment_key, $this->merchant_uuid);
		$this->lifetime = $this->get_option('lifetime') ?? 2;
		$this->theme = $this->get_option('theme') ?? 'light';
		$this->h2h = $this->get_option('h2h');
		$this->accepted_networks = $this->get_option('accepted_networks');
		$this->accepted_currencies = $this->get_option('accepted_currencies');

		$this->icon = plugin_dir_url(dirname(__FILE__)) . "images/logo.svg";
		if (!empty(get_option('heleket_method_image')))
		{
			$this->icon = esc_url(get_option('heleket_method_image'));
		}

		add_action("woocommerce_update_options_payment_gateways_{$this->id}", array($this, 'process_admin_options'));
	}

	/**
	 * @return array
	 */
	public function adminFields()
	{
		return [
			'enabled'             => [
				'title'    => esc_html__('Enabled', 'heleket-crypto-gateway-for-woocommerce'),
				'type'     => 'checkbox',
				'default'  => 'no',
				'desc_tip' => true
			],
			'merchant_uuid'       => [
				'title' => esc_html__('Merchant UUID', 'heleket-crypto-gateway-for-woocommerce'),
				'type'  => 'text'
			],
			'payment_key'         => [
				'title' => esc_html__('Payment API key', 'heleket-crypto-gateway-for-woocommerce'),
				'type'  => 'text'
			],
			'method_title'        => [
				'title'   => esc_html__('Method title', 'heleket-crypto-gateway-for-woocommerce'),
				'type'    => 'text',
				'default' => esc_html__('Pay with Heleket (USDT, USDC, BTC, ETH, LTC, and more)', 'heleket-crypto-gateway-for-woocommerce')
			],
			'description'         => [
				'title'   => esc_html__('Method description', 'heleket-crypto-gateway-for-woocommerce'),
				'type'    => 'text',
				'default' => esc_html__('Crypto payment system', 'heleket-crypto-gateway-for-woocommerce')
			],
			'subtract'            => [
				'title'             => esc_html__('How much commission does the client pay (0-100%)', 'heleket-crypto-gateway-for-woocommerce'),
				'description'       => esc_html__('Percentage of the payment commission charged to the client. If you have a rate of 1%, then if you create an invoice for 100 USDT with subtract = 100 (the client pays 100"%" commission), the client will have to pay 101 USDT.', 'heleket-crypto-gateway-for-woocommerce'),
				'type'              => 'number',
				'default'           => 0,
				'custom_attributes' => [
					'min' => 0,
					'max' => 100,
				],
			],
			'lifetime'            => [
				'title'   => esc_html__('Invoice Lifetime', 'heleket-crypto-gateway-for-woocommerce'),
				'type'    => 'select',
				'options' => [
					'1'  => '1 Hour',
					'2'  => '2 Hours',
					'3'  => '3 Hours',
					'4'  => '4 Hours',
					'5'  => '5 Hours',
					'6'  => '6 Hours',
					'7'  => '7 Hours',
					'8'  => '8 Hours',
					'9'  => '9 Hours',
					'10' => '10 Hours',
					'11' => '11 Hours',
					'12' => '12 Hours',
				],
				'default' => '2',
			],
			'h2h'                 => [
				'title'   => esc_html__('Host-to-Host', 'heleket-crypto-gateway-for-woocommerce'),
				'type'    => 'checkbox',
				'default' => 'no'
			],
			'theme'               => [
				'title'       => esc_html__('Theme', 'heleket-crypto-gateway-for-woocommerce'),
				'description' => esc_html__('If you select "Custom" theme the `/wp-content/plugins/heleket/templates/custom/*` template will be used. Change it carefully. Do not change names of the forms fields', 'heleket-crypto-gateway-for-woocommerce'),
				'type'        => 'select',
				'options'     => [
					'light'  => 'Light',
					'dark'   => 'Dark',
					'custom' => 'Custom',
				]
			],
			'accepted_networks'   => [
				'title'   => esc_html__('Accepted Networks', 'heleket-crypto-gateway-for-woocommerce'),
				'type'    => 'multiselect',
				'options' => [
					'ETH'       => 'Ethereum (ETH)',
					'ARBITRUM'  => 'Arbitrum (ARBITRUM)',
					'TRON'      => 'Tron (TRON)',
					'DOGE'      => 'Dogecoin (DOGE)',
					'DASH'      => 'Dash (DASH)',
					'AVALANCHE' => 'Avalanche (AVALANCHE)',
					'SOL'       => 'Solana (SOL)',
					'BTC'       => 'Bitcoin (BTC)',
					'XMR'       => 'Monero (XMR)',
					'TON'       => 'TON (TON)',
					'POLYGON'   => 'Polygon (POLYGON)',
					'BCH'       => 'Bitcoin Cash (BCH)',
					'LTC'       => 'Litecoin (LTC)',
					'BSC'       => 'Binance Smart Chain (BSC)',
				],
				'default' => [
					'ETH',
					'ARBITRUM',
					'TRON',
					'DOGE',
					'DASH',
					'AVALANCHE',
					'SOL',
					'BTC',
					'XMR',
					'TON',
					'POLYGON',
					'BCH',
					'LTC',
					'BSC',
				], // Выбраны все сети по умолчанию
				'class'   => 'wc-enhanced-select', // Добавим класс для стилизации
			],
			'accepted_currencies' => [
				'title'   => esc_html__('Accepted Currencies', 'heleket-crypto-gateway-for-woocommerce'),
				'type'    => 'multiselect',
				'options' => [
					'DOGE'  => 'Dogecoin (DOGE)',
					'SHIB'  => 'Shiba Inu (SHIB)',
					'CGPT'  => 'ChatGPT Token (CGPT)',
					'POL'   => 'POL (ex-MATIC)',
					'BCH'   => 'Bitcoin Cash (BCH)',
					'DAI'   => 'Dai (DAI)',
					'VERSE' => 'Verse (VERSE)',
					'SOL'   => 'Solana (SOL)',
					'BUSD'  => 'Binance USD (BUSD)',
					'LTC'   => 'Litecoin (LTC)',
					'ETH'   => 'Ethereum (ETH)',
					'BNB'   => 'Binance Coin (BNB)',
					'TRX'   => 'TRON (TRX)',
					'USDC'  => 'USD Coin (USDC)',
					'AVAX'  => 'Avalanche (AVAX)',
					'DASH'  => 'Dash (DASH)',
					'XMR'   => 'Monero (XMR)',
					'BTC'   => 'Bitcoin (BTC)',
					'USDT'  => 'Tether (USDT)',
					'TON'   => 'TON (TON)',
				],
				'default' => [
					'DOGE',
					'SHIB',
					'CGPT',
					'MATIC',
					'BCH',
					'DAI',
					'VERSE',
					'SOL',
					'BUSD',
					'LTC',
					'ETH',
					'BNB',
					'TRX',
					'USDC',
					'AVAX',
					'DASH',
					'XMR',
					'BTC',
					'USDT',
					'TON',
				], // Выбраны все валюты по умолчанию
				'class'   => 'wc-enhanced-select', // Добавляем класс для стилизации
			],
		];
	}

	/**
	 * @param $order_id
	 *
	 * @return array
	 */
	public function process_payment($order_id)
	{
		if ($this->h2h == "yes")
		{
			$order = wc_get_order($order_id);
			$order->update_status(PaymentStatus::WC_STATUS_PENDING);
			$order->save();
			wc_reduce_stock_levels($order_id);
			WC()->cart->empty_cart();

			return [
				'result'   => 'success',
				'redirect' => home_url('/heleket-pay?order_id=' . $order_id . '&step_id=1')
			];
		}
		else
		{
			$order = wc_get_order($order_id);
			$order->update_status(PaymentStatus::WC_STATUS_PENDING);
			$order->save();
			wc_reduce_stock_levels($order_id);
			WC()->cart->empty_cart();
			try
			{
				$success_url = $this->get_return_url($order);
				$return_url = str_replace('/order-received/', '/order-pay/', $success_url);
				$return_url .= '&pay_for_order=true';
				$payment = $this->payment->create([
					'amount'              => $order->get_total(),
					'currency'            => $order->get_currency(),
					'order_id'            => (string) $order_id,
					'url_return'          => $return_url,
					'url_success'         => $success_url,
					'url_callback'        => get_site_url(null, "wp-json/heleket-webhook/$this->merchant_uuid"),
					'is_payment_multiple' => true,
					'lifetime'            => (int) $this->lifetime * 3600,
					'subtract'            => $this->subtract,
				]);

				return ['result' => 'success', 'redirect' => $payment['url']];
			}
			catch (Exception $e)
			{
				$order->update_status(PaymentStatus::WC_STATUS_FAIL);
				wc_increase_stock_levels($order);
				$order->save();
			}

			return ['result' => 'success', 'redirect' => $this->get_return_url($order)];
		}
	}

	public function create_h2h_payment($order_id, $network, $to_currency)
	{
		$order = wc_get_order($order_id);
		try
		{
			$success_url = $this->get_return_url($order);
			$return_url = str_replace('/order-received/', '/order-pay/', $success_url);
			$return_url .= '&pay_for_order=true';

			$payment = $this->payment->create([
				'amount'              => $order->get_total(),
				'currency'            => $order->get_currency(),
				'order_id'            => 'woo_h2h_' . $network . '_' . $to_currency . '_' . (string) $order_id,
				'url_return'          => $return_url,
				'url_success'         => $success_url,
				'url_callback'        => get_site_url(null, "wp-json/heleket-webhook/$this->merchant_uuid"),
				'is_payment_multiple' => true,
				'lifetime'            => (int) $this->lifetime * 3600,
				'subtract'            => $this->subtract,
				'network'             => $network,
				'to_currency'         => $to_currency
			]);

			return $payment;
		}
		catch (Exception $e)
		{
			$order->update_status(PaymentStatus::WC_STATUS_FAIL);
			wc_increase_stock_levels($order);
			$order->save();

			return false;
		}
	}

	public function request_currencies()
	{
		$all_currencies = $this->payment->services();
		$filtered_currencies = [];
		foreach ($all_currencies as $currency)
		{
			if (
				$currency['is_available']
				&& in_array($currency['network'], $this->accepted_networks)
				&& in_array($currency['currency'], $this->accepted_currencies)
			)
			{
				array_push($filtered_currencies, $currency);
			}
		}

		return $filtered_currencies;
	}
}
