<?php
/**
 * The plugin bootstrap file
 *
 * This file is read by WordPress to generate the plugin information in the plugin
 * admin area. This file also includes all of the dependencies used by the plugin,
 * registers the activation and deactivation functions, and defines a function
 * that starts the plugin.
 *
 * @package           Heleket
 *
 * @wordpress-plugin
 * Plugin Name:       Heleket – Crypto Gateway for WooCommerce
 * Plugin URI:        https://wordpress.org/plugins/heleket-crypto-gateway-for-woocommerce/
 * Description:       Heleket payment plugin for WooCommerce.
 * Version:           1.3.5
 * Author:            TeslaCloud Studios
 * Author URI:        https://teslacloud.net
 * License: GPLv2
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:       heleket-crypto-gateway-for-woocommerce
 *
 * Requires Plugins: woocommerce
 * Requires at least: 6.3
 * Tested up to: 6.8
 * WC requires at least: 8.2
 * WC tested up to: 9.9.4
 */

use Automattic\WooCommerce\Utilities\FeaturesUtil;
use Heleket\Woocommerce\Gateway;
use Heleket\Woocommerce\PaymentStatus;

defined('ABSPATH') || exit;

require_once __DIR__ . '/vendor/autoload.php';

if (!in_array('woocommerce/woocommerce.php', apply_filters('active_plugins', get_option('active_plugins'))))
{
	return;
}

add_action('before_woocommerce_init', function ()
{
	if (class_exists(FeaturesUtil::class))
	{
		FeaturesUtil::declare_compatibility('custom_order_tables', __FILE__, true);
	}
});

add_filter('woocommerce_payment_gateways', function ($plugins)
{
	return array_merge([Gateway::class], $plugins);
});

add_filter('plugin_action_links_' . plugin_basename(__FILE__), function ($links)
{
	$url = admin_url('admin.php?page=wc-settings&tab=checkout&section=heleket-crypto-gateway-for-woocommerce');

	return array_merge(['<a href="' . $url . '">' . esc_html__('Configure', 'heleket-crypto-gateway-for-woocommerce') . '</a>'], $links);
});

add_action('plugins_loaded', function ()
{
	return new Gateway();
});

add_filter('wc_order_statuses', function ($statuses)
{
	$statuses['wc-wrong-amount'] = esc_html__('Wrong amount', 'heleket-crypto-gateway-for-woocommerce');

	return $statuses;
});

function heleket_add_endpoint()
{
	add_rewrite_endpoint('heleket-pay', EP_ROOT);
	flush_rewrite_rules(false);
}

add_action('init', 'heleket_add_endpoint');

function heleket_template_include($template)
{
	global $wp_query;
	if (isset($wp_query->query_vars['heleket-pay']))
	{
		$gateway = new Heleket\Woocommerce\Gateway();

		if ($gateway->h2h !== "yes")
		{
			wp_redirect(home_url());
			exit;
		}

		$order_id = isset($_GET['order_id']) ? intval($_GET['order_id']) : null;
		$step_id = isset($_GET['step_id']) ? intval($_GET['step_id']) : 1;
		$order = wc_get_order($order_id);
		$success_url = $gateway->get_return_url($order);
		$return_url = str_replace('/order-received/', '/order-pay/', $success_url);
		$return_url .= '&pay_for_order=true';

		$params = [
			'order_id'    => $order_id,
			'step_id'     => $step_id,
			'return_url'  => $return_url,
			'success_url' => $success_url,
			'theme'       => $gateway->theme,
		];

		switch ($step_id)
		{
			case 1:
				$params['currencies'] = $gateway->request_currencies();
				$params['unique_coins'] = array_unique(array_column($params['currencies'], 'currency'));
				sort($params['unique_coins']);
				$params['order_amount'] = $order->get_total();
				$params['order_currency'] = $order->get_currency();
				break;

			case 2:
				$network = sanitize_text_field(wp_unslash($_GET['network'] ?? null));
				$to_currency = sanitize_text_field(wp_unslash($_GET['to_currency'] ?? null));
				$params['payment'] = $gateway->create_h2h_payment($order_id, $network, $to_currency);
				$params['network'] = $network;
				$params['to_currency'] = $to_currency;
				break;

			default:
				break;
		}

		set_query_var('params', $params);
		$new_template = plugin_dir_path(__FILE__) . 'templates/' . ($params['theme'] === 'custom' ? 'custom/' : '') . 'form_' . $step_id . '.php';
		if (file_exists($new_template))
		{
			return $new_template;
		}
	}

	return $template;
}

add_filter('template_include', 'heleket_template_include');

function heleket_query_vars($vars)
{
	$vars[] = 'heleket-pay';

	return $vars;
}

add_filter('query_vars', 'heleket_query_vars');

function heleket_webhook_callback($request)
{
	$gateway = new Heleket\Woocommerce\Gateway();
	$params = $request->get_params();

	if (empty($params['uuid']))
	{
		return ['success' => false, 'error' => 'Missing required parameters'];
	}

	if(!empty($params['resend']))
	{
		$resendWebHook = $gateway->payment->reSendNotifications(['uuid' => sanitize_text_field($params['uuid'])]);

		return ['success' => true, 'error' => 'Webhook re-sent'];
	}

	// check signature
	if (!heleket_is_signature_valid($params))
	{
		return ['success' => false, 'error' => 'Invalid signature'];
	}

	$result = $gateway->payment->info(['uuid' => sanitize_text_field($params['uuid'])]);

	if (empty($result['payment_status']))
	{
		return ['success' => false, 'error' => 'Invalid payment status'];
	}

	$order_id = $result['order_id'];
	$order = wc_get_order($order_id);
	if (!$order)
	{
		return ['success' => false, 'error' => 'Order not found'];
	}

	$all_downloadable_and_virtual = heleket_all_items_downloadable_and_virtual($order);
	$order->set_status(PaymentStatus::convertToWoocommerceStatus($result['payment_status'], $all_downloadable_and_virtual));
	$order->save();

	if (PaymentStatus::isNeedReturnStocks($result['payment_status'], $all_downloadable_and_virtual))
	{
		wc_increase_stock_levels($order);
	}

	return ['success' => true];
}

function heleket_is_signature_valid(array &$data)
{
	$gateway_settings = get_option('woocommerce_heleket-crypto-gateway-for-woocommerce_settings');
	$apiKey = $gateway_settings['payment_key'];

	if (!$apiKey)
	{
		return false;
	}

	$signature = $data['sign'] ?? null;
	if (!$signature)
	{
		return false;
	}

	unset($data['sign']);

	$hash = md5(base64_encode(json_encode($data, JSON_UNESCAPED_UNICODE)) . $apiKey);
	if (!hash_equals($hash, $signature))
	{
		return false;
	}

	return true;
}

add_action('rest_api_init', function ()
{
	$gateway = new Heleket\Woocommerce\Gateway();
	register_rest_route('heleket-webhook', $gateway->merchant_uuid, [
		'methods'             => 'POST',
		'permission_callback' => '__return_true',
		'callback'            => 'heleket_webhook_callback',
	]);
});

add_action('rest_api_init', function ()
{
	register_rest_route('heleket-pay', '/check-status', [
		'methods'             => 'POST',
		'permission_callback' => '__return_true',
		'callback'            => 'heleket_check_payment_status',
	]);
});

function heleket_check_payment_status(WP_REST_Request $request)
{
	$order_id = $request->get_param('order_id');
	if (empty($order_id))
	{
		return new WP_REST_Response(['error' => 'Order ID is required'], 400);
	}
	$gateway = new Heleket\Woocommerce\Gateway();
	$result = $gateway->payment->info(['order_id' => $order_id]);
	$orderArray = explode('_', $order_id);

	$order = wc_get_order(end($orderArray));
	$all_downloadable_and_virtual = heleket_all_items_downloadable_and_virtual($order);

	$order->set_status(PaymentStatus::convertToWoocommerceStatus($result['payment_status'], $all_downloadable_and_virtual));
	$order->save();

	if (PaymentStatus::isNeedReturnStocks($result['payment_status'], $all_downloadable_and_virtual))
	{
		wc_increase_stock_levels($order);
	}

	return new WP_REST_Response($result, 200);
}

add_action('admin_enqueue_scripts', function ()
{
	if (
		isset($_GET['page'])
		&& sanitize_text_field($_GET['page']) === 'wc-settings'
		&& isset($_GET['tab'])
		&& sanitize_text_field($_GET['tab']) === 'checkout'
		&& isset($_GET['section'])
		&& sanitize_text_field($_GET['section']) === 'heleket-crypto-gateway-for-woocommerce'
	)
	{
		wp_enqueue_script(
			'plugin-admin-script',
			plugins_url('js/plugin-admin.js', __FILE__),
			['jquery'],
			time(),
			true
		);
	}
});

function heleket_all_items_downloadable_and_virtual($order)
{
	$items = $order->get_items();
	foreach ($items as $item)
	{
		$product = $item->get_product();
		if (!($product->is_virtual() && $product->is_downloadable()))
		{
			return false;
		}
	}

	return true;
}

function heleket_declare_cart_checkout_blocks_compatibility()
{
	// Check if the required class exists
	if (class_exists('\Automattic\WooCommerce\Utilities\FeaturesUtil'))
	{
		// Declare compatibility for 'cart_checkout_blocks'
		FeaturesUtil::declare_compatibility('cart_checkout_blocks', __FILE__, true);
	}
}

// Hook the custom function to the 'before_woocommerce_init' action
add_action('before_woocommerce_init', 'heleket_declare_cart_checkout_blocks_compatibility');

// Hook the custom function to the 'woocommerce_blocks_loaded' action
add_action('woocommerce_blocks_loaded', 'heleket_register_order_approval_payment_method_type');
/**
 * Custom function to register a payment method type
 */
function heleket_register_order_approval_payment_method_type()
{
	// Include the custom Blocks Checkout class
	require_once plugin_dir_path(__FILE__) . 'class-block.php';
	// Hook the registration function to the 'woocommerce_blocks_payment_method_type_registration' action
	add_action(
		'woocommerce_blocks_payment_method_type_registration',
		function (Automattic\WooCommerce\Blocks\Payments\PaymentMethodRegistry $payment_method_registry)
		{
			$payment_method_registry->register(new Heleket_Gateway_Blocks);
		}
	);
}
